% Example 1
%
% Worked example to explain the approach of timing recovery on
% pixel-by-pixel basis in sequential spatial data sets as displayed in
% Figure 2 of van Belzen et al 2017
%
% Reference:
% van Belzen, J, van de Koppel, J, van der Wal, D, Herman, P.M.J, Dakos, V,
% Kefi, S, Scheffer, M, Bouma, T.J "Timing recovery of ecosystems in 
% sequential remotely sensed and simulated data" Protocol Exchange (2017) 
% DOI: 10.1038/protex.2017.xxx
%
% Version 1.0 (25/07/2014)
% Version 1.1 (25/01/2017)
% Coded and developed by Jim van Belzen
% published under the Creative Commons Attribution Non-Commercial license 
% which allows users to read, copy, distribute and make derivative works 
% for noncommercial purposes from the material, as long as the author of 
% the original work is cited.
% This code comes with no warranties or support
% http://people.zeelandnet.nl/jbelzen/


%% Clear workspace and command  window
clear all
clc

on=1;
off=0;


%% Load & prepare example data
% Snapshots (grids)
t0 = dlmread('example_data/data_1/t0.grid');
t1 = dlmread('example_data/data_1/t1.grid');
t2 = dlmread('example_data/data_1/t2.grid');
t3 = dlmread('example_data/data_1/t3.grid');
t4 = dlmread('example_data/data_1/t4.grid');
t5 = dlmread('example_data/data_1/t5.grid');
t6 = dlmread('example_data/data_1/t6.grid');
t7 = dlmread('example_data/data_1/t7.grid');

% Concatenate snapshots [ 1.4) Stack grids ]
Seq = cat(3, t0, t1, t2, t3, t4, t5, t6, t7);

% Time of snapshots
time = [0, 2, 8, 19, 24, 30, 34, 36];


%% Calculate average recovery time, Tr 
% [ 2) Timing recovery in sequential spatial data: ]

[Tr] = fTimeRecov(Seq, time);


%% Graphical output
if on,
    
    % Get screen dimensions
    ScrS = get(0,'ScreenSize'); 
    ScrDim = ScrS(3:4);
    WinDim=floor(ScrDim.*[0.9 0.3]);

    %%- Color maps
    % CA color maps
    cmap1 = [1 1 1; 0 1 0];
    cmap2 = [1 1 1; 0 1 0; 1 0 0; 1 1 0];

    % Tr color map
    cmap3 = [0.75, 0.75, 0.75; 1 1 1;];
    cmap3(3:24,2) = linspace(1, 0.1, 22)';
    cmap3(3:24,3) = 0.35;
    
    %%- Sequence of snapshots
    WinDim=floor(ScrDim.*[0.9 0.8]);
    Figure1=figure('Position',[(ScrDim-WinDim)*0.75 WinDim],'color','w');

    subplot(2,4,1)
    pcolor(flipud([zeros(1,8); t0, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t0')

    subplot(2,4,2)
    pcolor(flipud([zeros(1,8); t1, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t1')

    subplot(2,4,3)
    pcolor(flipud([zeros(1,8); t2, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t2')

    subplot(2,4,4)
    pcolor(flipud([zeros(1,8); t3, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t3')

    subplot(2,4,5)
    pcolor(flipud([zeros(1,8); t4, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t4')

    subplot(2,4,6)
    pcolor(flipud([zeros(1,8); t5, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t5')

    subplot(2,4,7)
    pcolor(flipud([zeros(1,8); t6, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t6')

    subplot(2,4,8)
    pcolor(flipud([zeros(1,8); t7, zeros(7,1)]))
    colormap(cmap1)
    axis image
    title('t7')
    
    %%- Compare sequential snapshots
    WinDim=floor(ScrDim.*[0.9 0.8]);
    Figure2=figure('Position',[(ScrDim-WinDim)*0.75 WinDim],'color','w');
    
    T0 = t0;
    T1 = t1;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,1)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t1 - t0')
    
    T0 = t1;
    T1 = t2;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,2)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t2 - t1')
    
    T0 = t2;
    T1 = t3;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,3)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t3 - t2')
    
    T0 = t3;
    T1 = t4;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,4)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t4 - t3')
    
    T0 = t4;
    T1 = t5;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,5)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t5 - t4')
    
    T0 = t5;
    T1 = t6;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,6)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t6 - t5')
    
    T0 = t6;
    T1 = t7;
    diff = 1*(T0==1 & T1==1) + 2*(T0==1 & T1==0)+ 3*(T0==0 & T1==1);
    
    subplot(2,4,7)
    imagesc(diff,[0 4])
    colormap(cmap2)
    axis image
    title('t7 - t6')
    
    
    %%- Sequence of snapshots
    WinDim=floor(ScrDim.*[0.4 0.5]);
    Figure3=figure('Position',[(ScrDim-WinDim)*0.75 WinDim],'color','w');
    
    colormap(cmap3)
    pcolor(flipud([zeros(1,8); Tr, zeros(7,1)]))
    axis image
    colorbar
    title('Recovery time, Tr')

end